import express from 'express';
import { PKPass } from 'passkit-generator';
import path from 'path';
import { fileURLToPath } from 'url';
import fs from 'fs';

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

const app = express();

// Create passes directory if it doesn't exist
const passesDir = path.join(__dirname, 'passes');
if (!fs.existsSync(passesDir)) {
  fs.mkdirSync(passesDir);
}

// Helper function to generate a unique serial number
const generateSerialNumber = () => {
  return 'SN' + Date.now().toString(36).toUpperCase();
};

// Helper function to calculate warranty expiry date
const calculateWarrantyExpiry = (purchaseDate, warrantyMonths = 12) => {
  const date = new Date(purchaseDate);
  if (isNaN(date.getTime())) {
    throw new Error('Invalid purchase date format');
  }
  date.setMonth(date.getMonth() + warrantyMonths);
  return date.toISOString().split('T')[0];
};

// Helper function to validate and format date
const validateAndFormatDate = (dateStr) => {
  const date = new Date(dateStr);
  if (isNaN(date.getTime())) {
    throw new Error('Invalid date format');
  }
  return date.toISOString().split('T')[0];
};

app.get('/create-ios-pass', async (req, res) => {
  try {
    const {
      productName = 'Altherr',
      model = 'Altherr',
      purchaseDate = new Date().toISOString().split('T')[0],
      warrantyMonths = 12,
      warrantyType = 'Altherr',
      organizationName = 'Altherr',
    } = req.query;

    // Validate dates
    const validatedPurchaseDate = validateAndFormatDate(purchaseDate);
    const serialNumber = generateSerialNumber();
    const warrantyExpiry = calculateWarrantyExpiry(validatedPurchaseDate, warrantyMonths);

    // Log generated data for audit
    console.log('Generating pass with data:', {
      serialNumber,
      productName,
      model,
      purchaseDate: validatedPurchaseDate,
      warrantyExpiry,
      warrantyType,
      organizationName,
      timestamp: new Date().toISOString()
    });

    const wwdr = fs.readFileSync(path.join(__dirname, 'wwdr.pem'));
    const signerCert = fs.readFileSync(path.join(__dirname, 'passTypeCertificate.pem'));
    const signerKey = fs.readFileSync(path.join(__dirname, 'key-decrypted.pem'));

    const pass = await PKPass.from({
      model: path.join(__dirname, 'model.pass'),
      certificates: {
        wwdr,
        signerCert,
        signerKey
      },
    });

    // Set pass properties
    pass.serialNumber = serialNumber;
    pass.description = `${productName} Warranty Card`;
    pass.organizationName = organizationName;

    // Set dynamic fields
    pass.generic = {
      headerFields: [
        {
          key: 'productName',
          label: 'PRODUCT',
          value: productName
        }
      ],
      primaryFields: [
        {
          key: 'serialNumber',
          label: 'SERIAL NUMBER',
          value: serialNumber
        }
      ],
      secondaryFields: [
        {
          key: 'model',
          label: 'MODEL',
          value: model
        },
        {
          key: 'purchaseDate',
          label: 'PURCHASE DATE',
          value: validatedPurchaseDate
        }
      ],
      auxiliaryFields: [
        {
          key: 'warrantyExpiry',
          label: 'WARRANTY EXPIRES',
          value: warrantyExpiry
        },
        {
          key: 'warrantyType',
          label: 'WARRANTY TYPE',
          value: warrantyType
        }
      ]
    };

    // Update barcode with serial number
    pass.barcode = {
      message: serialNumber,
      format: 'PKBarcodeFormatQR',
      messageEncoding: 'iso-8859-1'
    };

    const stream = pass.getAsStream();
    
    // Save pass to disk
    const outPath = path.join(passesDir, `${serialNumber}.pkpass`);
    const writeStream = fs.createWriteStream(outPath);
    stream.pipe(writeStream);

    // Send pass to client
    res.type('application/vnd.apple.pkpass');
    stream.pipe(res);

    // Log successful generation
    writeStream.on('finish', () => {
      console.log(`Pass saved to: ${outPath}`);
    });

  } catch (err) {
    console.error('Error generating pass:', err);
    res.status(500).json({ 
      error: err.message || err.toString(),
      timestamp: new Date().toISOString()
    });
  }
});

const PORT = 3000;
app.listen(PORT, () => {
  console.log(
    `✅ Apple Wallet server running at http://localhost:${PORT}/create-ios-pass`,
  );
});

//http://localhost:3000/create-ios-pass?productName=iPhone&model=A2849&purchaseDate=2024-03-20&warrantyMonths=24&warrantyType=Extended