import * as Schemas from "./schemas/index.js";
import * as Utils from "./utils.js";
import * as Messages from "./messages.js";
/**
 * Class to represent lower-level keys pass fields
 * @see https://apple.co/2wkUBdh
 */
const passInstanceSymbol = Symbol("passInstance");
const sharedKeysPoolSymbol = Symbol("keysPool");
const fieldSchemaSymbol = Symbol("fieldSchema");
export default class FieldsArray extends Array {
    constructor(passInstance, keysPool, fieldSchema, ...args) {
        super(...args);
        this[fieldSchemaSymbol] = fieldSchema;
        this[passInstanceSymbol] = passInstance;
        this[sharedKeysPoolSymbol] = keysPool;
    }
    push(...items) {
        const validItems = registerWithValidation(this, ...items);
        return super.push(...validItems);
    }
    pop() {
        return unregisterItems(this, () => super.pop());
    }
    splice(start, deleteCount, ...items) {
        // Perfoming frozen check, validation and getting valid items
        const validItems = registerWithValidation(this, ...items);
        for (let i = start; i < start + deleteCount; i++) {
            this[sharedKeysPoolSymbol].delete(this[i].key);
        }
        return super.splice(start, deleteCount, ...validItems);
    }
    shift() {
        return unregisterItems(this, () => super.shift());
    }
    unshift(...items) {
        const validItems = registerWithValidation(this, ...items);
        return super.unshift(...validItems);
    }
}
function registerWithValidation(instance, ...items) {
    Utils.assertUnfrozen(instance[passInstanceSymbol]);
    let validItems = [];
    for (const field of items) {
        if (!field) {
            console.warn(Messages.format(Messages.FIELDS.INVALID, field));
            continue;
        }
        try {
            Schemas.assertValidity(instance[fieldSchemaSymbol], field, Messages.FIELDS.INVALID);
            if (instance[sharedKeysPoolSymbol].has(field.key)) {
                throw new TypeError(Messages.format(Messages.FIELDS.REPEATED_KEY, field.key));
            }
            instance[sharedKeysPoolSymbol].add(field.key);
            validItems.push(field);
        }
        catch (err) {
            if (err instanceof Error) {
                console.warn(err.message ? err.message : err);
            }
            else {
                console.warn(err);
            }
        }
    }
    return validItems;
}
function unregisterItems(instance, removeFn) {
    Utils.assertUnfrozen(instance[passInstanceSymbol]);
    const element = removeFn();
    instance[sharedKeysPoolSymbol].delete(element.key);
    return element;
}
//# sourceMappingURL=data:application/json;base64,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